<?php

namespace Solokod\SoloKargo;

use WC_Order;
use WC_Countries;

class Helper
{
    private WC_Order $order;
    private bool $isShipping = false; // Kargo icin fatura adresini kullan

    public function __construct(int $orderID)
    {
        $this->order = wc_get_order($orderID);

        // Kargo gönderim adresini ayarla
        $shippingAddress = $this->order->get_shipping_address_1();
        if (!empty($shippingAddress)) {
            $this->isShipping = true; // Kargo icin gonderim adresini kullan
        }
    }

    public function getCargoCode(int $orderID): string
    {
        // 24_10_005_000000
        $year = date('y'); // Yılın son 2 hanesi
        $month = date('m'); // Ay

        // wp_options tablosundan müşteri ID'sini al
        $customerId = get_option('solokargo_customer_id');

        // Sayı değilse veya boşsa varsayılan 0 olsun
        $customerId = intval($customerId);

        // 3 haneli hale getir (örnek: 5 -> 005)
        $customerCode = str_pad($customerId, 3, '0', STR_PAD_LEFT);

        // Kalan basamak sayısını hesapla
        $remainingDigits = 13 - strlen($year . $month . $customerCode);

        // orderID’yi kalan basamak kadar sıfırla doldur
        $orderNumber = str_pad($orderID, $remainingDigits, '0', STR_PAD_LEFT);

        // Sonuç: YYMM + MüşteriKodu(3hane) + orderID(0 dolu)
        $cargoKey = $year . $month . $customerCode . $orderNumber;

        return $cargoKey;
    }

    public function getCustomerName(): string
    {
        $first_name = $this->order->get_billing_first_name();
        $last_name = $this->order->get_billing_last_name();

        if ($this->isShipping) {
            $first_name = $this->order->get_shipping_first_name();
            $last_name = $this->order->get_shipping_last_name();
        }

        return implode(' ', array($first_name, $last_name));
    }

    public function getCustomerAddress(): string
    {
        $address = $this->order->get_address('billing');
        $shipAddress = array($address['address_1'], $address['address_2'], $address['postcode']);

        if ($this->isShipping) {
            $address = $this->order->get_address('shipping');
            $shipAddress = array($address['address_1'], $address['address_2'], $address['postcode']);
        }

        if ($shipAddress == '') {
            wc_get_logger()->info('Siparis no ' . $this->order->get_id() . ' için gönderim adresi boş geliyor');
        }

        return implode(' ', $shipAddress);
    }

    public function getCustomerCity(): string
    {
        $billing = $this->order->get_address('billing');
        $shipping = $this->order->get_address('shipping');

        /*
        $state = $this->isShipping ? $shipping['state'] : $billing['state'];
        $country = $this->isShipping ? $shipping['country'] : $billing['country'];

        $countries = new WC_Countries();
        $states = $countries->get_states($country);
        */

        $city = $this->isShipping ? $shipping['city'] : $billing['city'];
        error_log("States: " . var_export($city, true));

        return $city;
    }

    public function getCustomerTown(): string
    {
        $billing = $this->order->get_address('billing');
        $shipping = $this->order->get_address('shipping');

        return $this->isShipping ? $shipping['city'] : $billing['city'];
    }

    public function getCustomerEmail(): string
    {
        return $this->order->get_billing_email();
    }

    public function getCustomerPhone(): string
    {
        return $this->isShipping
            ? $this->order->get_shipping_phone()
            : $this->order->get_billing_phone();
    }

    public function getCargoBarcode(): string
    {
        return $this->order->get_meta('solokargo_cargo_barcode');
    }

    public function getFirstName(): string
    {
        $first_name = $this->order->get_billing_first_name();

        if ($this->isShipping) {
            $first_name = $this->order->get_shipping_first_name();
        }

        return $first_name;
    }

    public function getLastName(): string
    {
        $last_name = $this->order->get_billing_last_name();

        if ($this->isShipping) {
            $last_name = $this->order->get_shipping_last_name();
        }

        return $last_name;
    }

    public function getProductName(): string
    {
        $items = $this->order->get_items();
        $itemArray = array_values($items);
        return $itemArray[0]->get_name();
    }

    public function getCargoStatus(): string
    {
        return $this->order->get_meta('solokargo_cargo_status');
    }

    public function getPaymentMethod(): string
    {
        return $this->order->get_payment_method();
    }

    public function getOrderTotal(): string
    {
        return $this->order->get_total();
    }

    public function getItemsList(): array
    {
        $items = [];
        foreach ($this->order->get_items() as $item) {
            $p = $item->get_product();
            $sku = $p ? $p->get_sku() : '';

            $product = new \stdClass();
            $product->name = $item->get_name();
            $product->sku = $sku;
            $product->quantity = $item->get_quantity();
            $product->price = floatval($item->get_total());

            array_push($items, $product);
        }

        return $items;
    }
}
