<?php

namespace Solokod\SoloKargo\Modules;

use Solokod\SoloKargo\Config;
use Solokod\SoloKargo\Helper;
use Solokod\SoloKargo\Modules\KargoMail;
use Solokod\SoloKargo\Payment\GatewayBlocksSupport;
use WC_Order;
use Automattic\WooCommerce\Utilities\FeaturesUtil;
use Automattic\WooCommerce\Blocks\Payments\PaymentMethodRegistry;

class Startup
{
    public function __construct()
    {
        // actions
        add_action('init', [$this, 'registerCargoStatus']);
        add_action('admin_menu', [$this, 'registerMenu']);
        add_action("woocommerce_api_notification-hook", [$this, "notificationHook"]);

        // Ship To - Alıcı kolonuna siparis bilgise eklemek icin eski ve yeni sürümde
        add_action('manage_shop_order_posts_custom_column', [$this, 'orderTableAddDescription'], 20, 2);
        add_filter('manage_woocommerce_page_wc-orders_custom_column', [$this, 'orderTableAddDescription'], 20, 2);

        // filters
        add_filter('wc_order_statuses', [$this, 'cargoStatusAddStatusList']);
        add_filter('woocommerce_email_classes', [$this, 'registerEmailClasses']);

        // Kapida odeme icin kredi karti odeme metodu ekler
        add_filter('woocommerce_payment_gateways', [$this, 'addGatewayClass']);
        add_action('woocommerce_blocks_loaded', [$this, 'gatewayBlockSupport']);
        add_action('before_woocommerce_init', [$this, 'checkoutBlocksCompatibility']);

        // Musteri hesabım sayfası siparis tablosuna takip kodu ekler
        add_filter('woocommerce_my_account_my_orders_column_order-status', [$this, 'addCargoTrackingCode']);

        // test
        //add_filter('manage_woocommerce_page_wc-orders_columns', [$this, 'shippingAddressClear'], 10, 1);
        //add_action('woocommerce_shop_order_list_table_custom_column', [$this, 'orderListAddDescription'], 10, 2);

        // Add wc-orders bulk action item and bulk barcode item
        add_filter('bulk_actions-woocommerce_page_wc-orders', [$this, 'addBulkActionCustomItem'], 20, 1);
        // add_filter('handle_bulk_actions-woocommerce_page_wc-orders', [$this, 'handleBulkActionShipmentItem'], 20, 3);

        add_action('wp_ajax_getOrderModalData', [$this, 'getOrderModalData']);
        add_action('wp_ajax_getBulkBarcodeData', [$this, 'getBulkBarcodeData']);
    }

    function getBulkBarcodeData(): void
    {
        if (isset($_POST['order_ids'])) {
            $order_ids = json_decode(stripslashes($_POST['order_ids']));  // JSON string olarak gelen veriyi diziye çevir

            error_log(print_r($order_ids, true));

            if (is_array($order_ids)) {
                $orderIDs = implode(",", $order_ids);   // OrderID degelerini virgul ile ayrilmis listeye cevir

                ob_start();
                require SOLOKARGO_DIR . 'includes/templates/barcodes/barcode-print.php';
                $html = ob_get_clean();
        
                $response = [
                    "status" => true,
                    "message" => "",
                    "data" => $html
                ];

                error_log(print_r($response, true));

                wp_send_json_success($response);  // Coklu barkod verisini başarılı şekilde geri döndür

            } else {
                wp_send_json_error('Geçersiz sipariş ID\'leri');
            }
        }

        wp_die();  // İşlem tamamlandı
    }

    function getOrderModalData(): void
    {
        if (isset($_POST['order_ids'])) {
            $order_ids = json_decode(stripslashes($_POST['order_ids']));  // JSON string olarak gelen veriyi diziye çevir

            if (is_array($order_ids)) {
                $ordersData = [];

                foreach ($order_ids as $order_id) {
                    $order = wc_get_order($order_id);  // Sipariş objesini al

                    $cargo_key = $order->get_meta('solokargo_cargo_key');

                    if ($order) {
                        $orderData = [
                            'id' => $order->get_id(),
                            'status' => $order->get_status(),
                            'isSend' => !empty($cargo_key),
                            'total' => $order->get_total(),
                            'customer' => $order->get_billing_first_name() . ' ' . $order->get_billing_last_name(),
                            'dateCreated' => $order->get_date_created()->format('Y-m-d H:i:s'),
                            'defaultFirm' => get_option('solokargo_default_firm'),
                            'defaultFirmName' => Config::getFirmName(get_option('solokargo_default_firm')) . ' Kargo'
                        ];
                        $ordersData[] = $orderData;  // Her siparişi veriler dizisine ekle
                    }
                }

                wp_send_json_success($ordersData);  // Çoklu sipariş verisini başarılı şekilde geri döndür
            } else {
                wp_send_json_error('Geçersiz sipariş ID\'leri');
            }
        }
        wp_die();  // İşlem tamamlandı
    }

    public function addBulkActionCustomItem(array $bulkActions): array
    {
        $newActions = [];
        foreach ($bulkActions as $key => $action) {
            $newActions[$key] = $action;
            if ('mark_cancelled' === $key) {
                $newActions['solokargo_bulk-shipment'] = __('SoloKargo - Toplu Kargo Gönderimi', 'solokargo');
                $newActions['solokargo_bulk-barcode'] = __('SoloKargo - Toplu Barkod Hazırla', 'solokargo');
            }
        }
        return $newActions;
    }

    public function addCargoTrackingCode($order)
    {
        $status = wc_get_order_status_name($order->get_status());
        $firmName = Config::getFirmName($order->get_meta('solokargo_selected_firm'));
        $trackingNo = $order->get_meta('solokargo_tracking_code');

        if ($status == 'Kargoya Verildi') {
            echo "$status<br>$firmName Kargo <span class='dashicons dashicons-location'></span><br>$trackingNo";
        } else {
            echo "<span>$status</span>";
        }
    }

    public function checkoutBlocksCompatibility()
    {
        // true (compatible, default) or false (not compatible)
        FeaturesUtil::declare_compatibility('cart_checkout_blocks', SOLOKARGO_DIR, true);
    }

    public function gatewayBlockSupport()
    {
        // Check if the required class exists
        if (!class_exists('Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType')) {
            return;
        }

        // Include the custom Blocks Checkout class
        require_once SOLOKARGO_DIR . '/includes/Payment/GatewayBlocksSupport.php';

        // Hook the registration function to the 'woocommerce_blocks_payment_method_type_registration' action
        add_action(
            'woocommerce_blocks_payment_method_type_registration',
            function (PaymentMethodRegistry $payment_method_registry) {
                // Register an instance of GatewayBlocksSupport
                $payment_method_registry->register(new GatewayBlocksSupport);
            }
        );
    }

    public function notificationHook()
    {
        $requiredFields = ['licenceKey', 'orderID', 'cargoStatus', 'trackingNo'];

        $data = $this->checkNotificationFields($requiredFields);

        // Verilere erişim sağlayın
        $licenceKey = $data['licenceKey'];
        $orderID = $data['orderID'];
        $cargoStatus = $data['cargoStatus'];
        $trackingNo = $data['trackingNo'];

        $logger = wc_get_logger();
        $logger->log('info', 'Notification Hook Data', $data);

        $order = wc_get_order($orderID);
        if ($order === false) {
            $logger->log('error', 'Order nesnesi null veya boş', $data);
            wp_send_json_error("Order nesnesi null veya bos. OrderID: $orderID");
            return;   // Fonksiyonu sonlandır
        }

        // Kargo bilgilerini siparise ekle
        $order->update_meta_data('solokargo_cargo_status', $cargoStatus);
        $order->update_meta_data('solokargo_tracking_code', $trackingNo);
        $order->save();

        // Kargo durumunu güncelle
        $order->update_status('sk-kargolandi');

        // Eposta gonderim durumu nedir?
        $isSendMail = $order->get_meta('solokargo_send_mail');

        if ($isSendMail == 0 && $trackingNo != '') {
            // Eposta gönder
            $kargoMail = WC()->mailer()->get_emails();
            $kargoMail['KargoMail']->trigger($orderID);

            $date = new \DateTime();
            error_log("Eposta gönderimi yapıldı..." . $date->format('l, d F Y H:i:s'));

            $order->update_meta_data('solokargo_send_mail', true);
            $order->save();
        }

        if ($cargoStatus === 'Teslim Edildi') {
            // Kargo durumunu güncelle
            $order->update_status('wc-completed');
        }

        wp_send_json_success('OK');
    }

    public function registerCargoStatus()
    {
        $new_status = [
            'label' => _x('Paketlendi', 'solokargo'),
            'public' => true,
            'exclude_from_search' => false,
            'show_in_admin_all_list' => true,
            'show_in_admin_status_list' => true,
            'label_count' => _n_noop('Paketlendi (%s)', 'Paketlendi (%s)'),
        ];
        register_post_status('wc-sk-paketlendi', $new_status);

        $new_status = [
            'label' => _x('Kargoya Verildi', 'solokargo'),
            'public' => true,
            'exclude_from_search' => false,
            'show_in_admin_all_list' => true,
            'show_in_admin_status_list' => true,
            'label_count' => _n_noop('Kargoya Verildi (%s)', 'Kargoya Verildi (%s)'),
        ];
        register_post_status('wc-sk-kargolandi', $new_status);
    }

    public function cargoStatusAddStatusList($order_statuses)
    {
        $new_order_statuses = [];
        foreach ($order_statuses as $key => $status) {
            $new_order_statuses[$key] = $status;
            if ('wc-on-hold' === $key) {
                $new_order_statuses['wc-sk-paketlendi'] = _x('Paketlendi', 'solokargo');
                $new_order_statuses['wc-sk-kargolandi'] = _x('Kargoya Verildi', 'solokargo');
            }
        }

        return $new_order_statuses;
    }

    public function orderTableAddDescription(string $column, $order_or_order_id): void
    {
        if ($column === 'shipping_address') {
            $order = $order_or_order_id instanceof WC_Order ? $order_or_order_id : wc_get_order($order_or_order_id);
            $firm = $order->get_meta('solokargo_selected_firm');

            if ($firm) {
                $helper = new Helper($order->get_id());
                $cargoStatus = $helper->getCargoStatus();

                $firmName = Config::getFirmName($firm);

                $msg = sprintf('%s: %s', $firmName, $cargoStatus);

                echo '<span class="description">' . esc_html($msg) . '</span>';
            } else {
                echo '<span class="description">Kargolama bekliyor!</span>';
            }
        }
    }

    public function addGatewayClass(array $gateways)
    {
        $gateways[] = 'Solokod\SoloKargo\Payment\KapidaOdemeKart';
        return $gateways;
    }

    public function registerMenu()
    {
        $page_title = __("Kargo Yönetim", "solokargo");
        $menu_title = __("Kargo", "solokargo");
        $capability = "manage_options";
        $menu_slug = "solokargo";
        $icon_url = "data:image/svg+xml;base64," . base64_encode('<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 640 512"><!--!Font Awesome Free 6.5.1 by @fontawesome - https://fontawesome.com License - https://fontawesome.com/license/free Copyright 2024 Fonticons, Inc.--><path fill="#9ca2a7" d="M48 0C21.5 0 0 21.5 0 48V368c0 26.5 21.5 48 48 48H64c0 53 43 96 96 96s96-43 96-96H384c0 53 43 96 96 96s96-43 96-96h32c17.7 0 32-14.3 32-32s-14.3-32-32-32V288 256 237.3c0-17-6.7-33.3-18.7-45.3L512 114.7c-12-12-28.3-18.7-45.3-18.7H416V48c0-26.5-21.5-48-48-48H48zM416 160h50.7L544 237.3V256H416V160zM112 416a48 48 0 1 1 96 0 48 48 0 1 1 -96 0zm368-48a48 48 0 1 1 0 96 48 48 0 1 1 0-96z"/></svg>');
        $position = 58;
        add_menu_page($page_title, $menu_title, $capability, $menu_slug, function () {}, $icon_url, $position);

        // Baslangic menusu
        $this->registerDashboard($menu_slug);

        // Kargo firma menuleri
        $options = get_option('solokargo_licence_info');
        if (!empty($options)) {
            $data = json_decode($options, true);
            $firms = explode(',', $data['subscribeFirm']);

            foreach ($firms as $firm) {
                if ($firm == 'surat') {
                    $this->registerFirmSurat($menu_slug);
                }

                if ($firm == 'yurtici') {
                    $this->registerFirmYurtici($menu_slug);
                }

                if ($firm == 'aras') {
                    $this->registerFirmAras($menu_slug);
                }

                if ($firm == 'mng') {
                    $this->registerFirmMng($menu_slug);
                }

                if ($firm == 'test') {
                    $this->registerFirmTest($menu_slug);
                }
            }
        }
    }

    public function registerDashboard(string $menuSlug)
    {
        $pageTitle = __('Genel ayarlar', 'solokargo');
        $menuTitle = __('Genel ayarlar', 'solokargo');
        $capability = "manage_options";
        $callback = function () {
            echo '<div id="solokargoDashboard"><h2>Loading...</h2></div>';
        };

        add_submenu_page($menuSlug, $pageTitle, $menuTitle, $capability, 'solokargo', $callback);
    }

    public function registerFirmSurat(string $menuSlug)
    {
        $pageTitle = __('Sürat Ayarlar', 'solokargo');
        $menuTitle = __('Sürat bilgiler', 'solokargo');
        $capability = "manage_options";
        $callback = function () {
            echo '<div id="solokargoSettingsSurat"><h2>Loading...</h2></div>';
        };

        add_submenu_page($menuSlug, $pageTitle, $menuTitle, $capability, 'solokargo-surat', $callback);
    }

    public function registerFirmYurtici(string $menuSlug)
    {
        $pageTitle = __('Yurtiçi kargo ayarları', 'solokargo');
        $menuTitle = __('Yurtiçi bilgiler', 'solokargo');
        $capability = "manage_options";
        $callback = function () {
            echo '<div id="solokargoSettingsYurtici"><h2>Loading...</h2></div>';
        };

        add_submenu_page($menuSlug, $pageTitle, $menuTitle, $capability, 'solokargo-yurtici', $callback);
    }

    public function registerFirmAras(string $menuSlug)
    {
        $pageTitle = __('Aras Kargo Bilgiler', 'solokargo');
        $menuTitle = __('Aras kargo ayarları', 'solokargo');
        $capability = "manage_options";
        $callback = function () {
            echo '<div id="solokargoSettingsAras"><h2>Loading...</h2></div>';
        };

        add_submenu_page($menuSlug, $pageTitle, $menuTitle, $capability, 'solokargo-aras', $callback);
    }

    public function registerFirmMng(string $menuSlug)
    {
        $pageTitle = __('MNG Kargo Bilgiler', 'solokargo');
        $menuTitle = __('MNG kargo ayarları', 'solokargo');
        $capability = "manage_options";
        $callback = function () {
            echo '<div id="solokargoSettingsMng"><h2>Loading...</h2></div>';
        };

        add_submenu_page($menuSlug, $pageTitle, $menuTitle, $capability, 'solokargo-mng', $callback);
    }

    public function registerFirmTest(string $menuSlug)
    {
        $pageTitle = __('Test Karg Bilgiler', 'solokargo');
        $menuTitle = __('Test kargo ayarları', 'solokargo');
        $capability = "manage_options";
        $callback = function () {
            echo '<div id="solokargoSettingsTest"><h2>Loading...</h2></div>';
        };

        add_submenu_page($menuSlug, $pageTitle, $menuTitle, $capability, 'solokargo-test', $callback);
    }

    public function registerEmailClasses($emailClasses)
    {
        $emailClasses['KargoMail'] = new KargoMail();
        return $emailClasses;
    }

    private function checkNotificationFields(array $requiredFields): array
    {
        $missingFields = array();

        foreach ($requiredFields as $field) {
            if (!isset($_POST[$field])) {
                $missingFields[] = $field;
            }
        }

        if (!empty($missingFields)) {
            $errorMessage = implode(', ', $missingFields) . ' alaninda bir hata olustu.';
            wp_send_json_error($errorMessage, 400);
            exit;
        }

        $data = [];
        foreach ($requiredFields as $field) {
            $data[$field] = $_POST[$field];
        }

        return $data;
    }
}
