<?php

namespace Solokod\SoloKargo\Services;

use Solokod\SoloKargo\Config;
use Solokod\SoloKargo\Helper;
use Solokod\SoloKargo\KargoApi;
use WP_REST_Controller;
use WP_REST_Request;
use WP_REST_Response;

class BoxService extends WP_REST_Controller
{
    private KargoApi $kargoApi;

    public function __construct(KargoApi $kargoApi)
    {
        add_action('rest_api_init', [$this, 'registerRoutes']);
        $this->kargoApi = $kargoApi;
    }

    public function registerRoutes()
    {
        $namespace = 'solokargo/v1';

        register_rest_route(
            $namespace,
            '/metabox/entry',
            [
                [
                    'methods' => 'GET',
                    'callback' => [$this, 'getCargoEntryParams'],
                    'permission_callback' => '__return_true',
                ],
                [
                    'methods' => 'POST',
                    'callback' => [$this, 'createCargoParams'],
                    'permission_callback' => '__return_true',
                ]
            ]
        );
        register_rest_route(
            $namespace,
            '/metabox/detail',
            [
                [
                    'methods' => 'GET',
                    'callback' => [$this, 'getCargoInfo'],
                    'permission_callback' => '__return_true',
                ],
                [
                    'methods' => 'DELETE',
                    'callback' => [$this, 'cancelCargoInfo'],
                    'permission_callback' => '__return_true',
                ]
            ]
        );
        register_rest_route(
            $namespace,
            '/metabox/cargo-info',
            [
                [
                    'methods' => 'GET',
                    'callback' => [$this, 'queryCargoParams'],
                    'permission_callback' => '__return_true',
                ]
            ]
        );

        register_rest_route(
            $namespace,
            '/metabox/print',
            [
                [
                    'methods' => 'GET',
                    'callback' => [$this, 'getCargoPrintInfo'],
                    'permission_callback' => '__return_true',
                ]
            ]
        );

    }

    public function getCargoPrintInfo(WP_REST_Request $request): WP_REST_Response
    {
        $orderIDs = $request['id'];

        ob_start();
        require SOLOKARGO_DIR . 'includes/templates/barcodes/barcode-print.php';
        $html = ob_get_clean();

        $response = [
            "status" => true,
            "message" => "",
            "data" => $html
        ];

        return $this->rezponse($response);
    }

    public function getCargoEntryParams(WP_REST_Request $request): WP_REST_Response
    {
        $version = Config::getVersion();

        $options = get_option('solokargo_licence_info');
        $data = json_decode($options, true);
        $firms = explode(',', $data['subscribeFirm']);

        $firmList = [];
        foreach ($firms as $f) {
            $data = [
                'firmCode' => $f,
                'firmName' => Config::getFirmName($f)
            ];

            $firmList[] = $data;
        }

        $result = [
            'firms' => $firmList,
            'version' => $version,
        ];

        return $this->rezponse($result);
    }

    public function getCargoInfo(WP_REST_Request $request): WP_REST_Response
    {
        $orderID = $request['id'];
        $isApiQuery = $request['query'];
        $status = true;
        $message = '';

        if ($isApiQuery === 'true') {
            $result = $this->getCargoInfoFromAPI($orderID);
            $status = $result['status'];
            $message = $result['message'];

            error_log(print_r($result, true));
        }

        $order = wc_get_order($orderID);

        $result = [];
        $result['status'] = $status;
        $result['message'] = $message;
        $result['version'] = Config::getVersion();
        $result['selectedFirm'] = $order->get_meta('solokargo_selected_firm');
        $result['cargoStatus'] = $order->get_meta('solokargo_cargo_status');
        $result['barcode'] = $order->get_meta('solokargo_cargo_key');
        $result['trackingCode'] = $order->get_meta('solokargo_tracking_code');

        error_log(print_r($result, true));

        return $this->rezponse($result);
    }

    public function createCargoParams(WP_REST_Request $request): WP_REST_Response
    {
        $params = $request->get_body_params();
        $orderID = $params['orderID'];
        $firmCode = $params['firm'];

        $licenceInfo = json_decode(get_option('solokargo_licence_info'), true);
        $data = json_decode(get_option("solokargo_{$firmCode}_info"), true);

        $helper = new Helper($orderID);
        $data['cargoKey'] = $helper->getCargoCode($orderID);
        $data['barcode'] = $helper->getCargoCode($orderID);
        $data['invoiceKey'] = $params['invoiceNo'];
        $data['cashOnDelivery'] = $helper->getPaymentMethod();
        $data['orderTotal'] = $helper->getOrderTotal();

        $cargoInfo = [
            "site" => $licenceInfo['site'],
            "licenceKey" => $licenceInfo['licenceKey'],
            "firmCode" => $firmCode,
            "orderID" => $params['orderID'],
            "callbackURL" => $this->getCallbackURL(),
            "data" => $data,
            "customer" => [
                "firstName" => $helper->getFirstName(),
                "lastName" => $helper->getLastName(),
                "company" => "",
                "address_1" => $helper->getCustomerAddress(),
                "address_2" => "",
                "state" => $helper->getCustomerTown(),
                "city" => $helper->getCustomerCity(),
                "email" => $helper->getCustomerEmail(),
                "phone" => $helper->getCustomerPhone()
            ],
            "products" => $helper->getItemsList()
        ];

        $response = $this->kargoApi->createCargo($cargoInfo);

        if ($response['status']) {
            $order = wc_get_order($params['orderID']);
            $order->update_meta_data('solokargo_selected_firm', $response['firmCode']);
            $order->update_meta_data('solokargo_cargo_key', $response['cargoKey']);
            $order->update_meta_data('solokargo_cargo_barcode', $response['barcode']);
            $order->update_meta_data('solokargo_cargo_status', 'Paketlendi');
            $order->update_meta_data('solokargo_invoice_number', $response['invoiceKey']);
            $order->update_meta_data('solokargo_tracking_code', '---');
            $order->save();

            $order->update_status('sk-paketlendi');
        } else {
            error_log(print_r($cargoInfo, true));
        }

        return $this->rezponse($response);
    }

    public function cancelCargoInfo(WP_REST_Request $request): WP_REST_Response
    {
        $order = wc_get_order($request['id']);
        $licenceInfo = json_decode(get_option('solokargo_licence_info'), true);

        $firmCode = $order->get_meta('solokargo_selected_firm');
        $firmOptions = json_decode(get_option("solokargo_{$firmCode}_info"), true);

        $params = [
            'site' => $licenceInfo['site'],
            'licenceKey' => $licenceInfo['licenceKey'],
            'firmCode' => $firmCode,
            'orderID' => strval($order->get_id()),
            'cargoKey' => $order->get_meta('solokargo_cargo_key'),
            'workMode' => $firmOptions['workMode'],
            'credentials' => $firmOptions['credentials']
        ];

        $response = $this->kargoApi->cancelCargo($params);
        if ($response['status']) {
            $order = wc_get_order($params['orderID']);
            $order->delete_meta_data('solokargo_selected_firm');
            $order->delete_meta_data('solokargo_cargo_key');
            $order->delete_meta_data('solokargo_cargo_barcode');
            $order->delete_meta_data('solokargo_cargo_status');
            $order->delete_meta_data('solokargo_invoice_number');
            $order->delete_meta_data('solokargo_tracking_code');
            $order->save();

            $order->update_status('on-hold');
        }

        return $this->rezponse($response);
    }

    private function getCallbackURL(): string
    {
        $protocol = "http://";
        if (isset($_SERVER["HTTPS"]) && $_SERVER["HTTPS"] === "on") {
            $protocol = "https://";
        }

        // https://wp.solokod.com/wc-api/notification-hook
        return $protocol . $_SERVER["HTTP_HOST"] . '/wc-api/notification-hook';
    }

    private function getCargoInfoFromAPI(string $orderID): array
    {
        $order = wc_get_order($orderID);

        $firmCode = $order->get_meta('solokargo_selected_firm');
        $licenceInfo = json_decode(get_option('solokargo_licence_info'), true);
        $firmOptions = json_decode(get_option("solokargo_{$firmCode}_info"), true);

        $params = [
            'site' => $licenceInfo['site'],
            'licenceKey' => $licenceInfo['licenceKey'],
            'firmCode' => $firmCode,
            'orderID' => strval($order->get_id()),
            'cargoKey' => $order->get_meta('solokargo_cargo_key'),
            'workMode' => $firmOptions['workMode'],
            'credentials' => $firmOptions['credentials']
        ];

        // Guncel kargo bilgilerini API uzerinden al ve siparisi guncelle
        $response = $this->kargoApi->queryCargo($params);
        if ($response['status']) {
            $trackingNo = $response['data']['trackingNo'] === '' ? '---' : $response['data']['trackingNo'];

            $order->update_meta_data('solokargo_cargo_status', $response['data']['cargoStatus']);
            $order->update_meta_data('solokargo_tracking_code', $trackingNo);
            $order->save();

            // Kargo durumunu güncelle
            $order->update_status('sk-kargolandi');
        }

        return [
            'status' => $response['status'],
            'message' => $response['message']
        ];
    }

    private function rezponse(array $data): WP_REST_Response
    {
        $response = new WP_REST_Response($data, 200);
        $response->header("Content-type", "application/json; charset=utf-8");

        return $response;
    }
}
